<?php
//
// Copyright (c) 2014-2015 NetApp, Inc.
// All rights reserved.
//
// VERSION="1.0"
//
// This validation script checks for 'supported platform' in a software release.
// When a new Data ONTAP software is released, some board families or system
// models may be removed from the supported platform list.  When this happens,
// the 'supported_systems' list below needs to be updated.
//
// IMPORTANT: Please ensure the 'supported_systems' array is updated for
//            EVERY SOFTWARE RELEASE.
//

$supported_systems = array (
    // Carnegie family (SB_XVI) supported model
    'XVI' =>   'FAS3270, FAS3220, FAS3250,'
             . 'V3270, V3220, V3250,'
             . 'SA320',
    // Absolut family (SB_XVIII) supported model
    'XVIII' => 'FAS6210, FAS6220, FAS6240, FAS6250, FAS6280, FAS6290,'
             . 'V6210, V6220, V6240, V6250, V6280, V6290,'
             . 'SA620, SA620A',
    // ATT, Vespa, Ruckus family (SB_XIX) supported model
    'XIX' =>   'FAS2554, FAS2552, FAS2520,'
             . 'FAS2240-2, FAS2240-4, Palomar,'
             . 'FAS2220',
    // BBA family (SB_XX) supported model
    'XX' =>    'FAS8020, FAS8040, FAS8060, FAS8080,'
	     . 'AFF8020, AFF8040, AFF8060, AFF8080',
    // LML family (SB_XXI) supported model
    'XXI' =>   'MPOC, LML, FASLODI, FASMDRA, FASLVMR,'
             . 'FAS9040, FAS9060, FAS9080, FAS9000,'
	     . 'AFF9040, AFF9060, AFF9080, AFF-A700',
    // Razor family (SB_XXII) supported model
    'XXII' =>  'Razor, FASRAZM, FASRAZL, FASRAZXL,'
             . 'FAS2950, FAS2980, FAS2600, FAS8200,'
             . 'FAS2650, FAS2620,'
             . 'AFF-A300, AFF-A200',
    // Aurora family (SB_XXIII) supported model
    'XXIII' => 'AFF-A700s',
);


//
// Messages issued to user/customer.
//
$action_msg = "Use the \"cluster unjoin\" command to remove the unsupported system "
            . "from the cluster or upgrade the controller hardware before "
            . "installing the new software release.";
$advice_msg = "Remove the unsupported system from the cluster or upgrade the "
            . "controller hardware before installing the new software release.";
$error_msg = "The following nodes are not supported: ";


$vsim_family = 'NetApp VSim';
$vsa_family = 'Virtual Machine';
$error_detected = false;
$token_seperator = " ,;\n\t";
$node_list = "";


function logDebugMessage($message, $log_error) {
    $dbgquery = "CALL add_scripting_log_message('validation_rules_platform_check: $message',
                 $log_error)";
    $result = dotsql_query($dbgquery);
}


/* Get board family and system model from SMF table 'system_node_info' */
$query = "SELECT system-name, system-machine-type, board-type, system-model FROM system_node_info";
$result = dotsql_query($query);

/* Perform platform support validation for all nodes found in a cluster */
while ($row = dotsql_fetch_object($result)) {
    $node_name = $row->{'system-name'};
    $board_family = $row->{'board-type'};
    $system_model = $row->{'system-model'};
    $node_list .= "$node_name/$board_family/$system_model ";

    if (strcasecmp($board_family, $vsim_family) === 0) {
        /* do not perform validation check for VSIM */
        continue;
    } elseif (strcasecmp($board_family, $vsa_family) === 0) {
        /* do not perform validation check for VSA */
        continue;
    } else {
        /* Extract the board family at the end of the 'board_type' string */
        if (($extracted_family = strrchr($board_family, ' ')) !== false) {
            $board_family = $extracted_family;
            $board_family = ltrim($board_family);
        }

        /* perform 'platform support validation' on actual hardware */
        if (array_key_exists($board_family, $supported_systems)) {
            /* supported board family */
            $tokenstr = strtok($supported_systems[$board_family], $token_seperator);

            /* check supported system models */
            while ($tokenstr !== false) {
                if (strcasecmp($system_model, $tokenstr) === 0) {
                    /* system model found in 'supported platform' list */
                    break;
                }
                $tokenstr = strtok($token_seperator);
            }

            if ($tokenstr === false) {
                /* did not found matched system model */
                if ($error_detected) {
                    $error_msg .= ", ";
                }
                $error_detected = true;
                $error_msg .= "$node_name ($system_model)";
            }
        } else {
            /* unsupported board family */
            if ($error_detected) {
                 $error_msg .= ", ";
            }
            $error_detected = true;
            $error_msg .= "$node_name ($system_model)";
        }
    }
}

if ($error_detected) {
    $error_msg .= ".";
    echo "<Alert>$error_msg</Alert>";
    echo "<Action>$action_msg</Action>";
    echo "<Advice>$advice_msg</Advice>";
    echo "<RC>error</RC>";
}
logDebugMessage("Nodes checked: $node_list", 'false');

exit;
?>
