#!/bin/ash
#
#
#  Update a persistent counter as a uBoot environment variable.
#

usage() {
    echo "Usage:  $0 <uBoot Environment Variable> <Count>"
    echo ""
    echo "           <uBoot Environment Variable> is the uBoot env vrbl name "
    echo "                  currently supported environment variables are:   "
    echo "                      supfpga_crcerr_bootcount,                    "
    echo "                      supfpga_crcerr_bootfails,                    "
    echo "                      supfpga_crcerr_runtimecount                  "
    echo "           <Count> is the increment (integer) to add to the vrbl value  "
    exit 1
}

#
# Ensure two parameters are provided.
#
if [ -z "$1" ] || [ -z "$2" ]; then
        echo "Must provide two parameters."
        usage
        exit 1
fi

#
# To prevent pollution of the uBoot environment variable space,
# only accept expected variable names.
#
case $1 in
	supfpga_crcerr_runtimecount)
		echo "Updating supfpga_crcerr_runtimecount"
		;;
	nvramfpga_crcerr_runtimecount)
		echo "Updating nvramfpga_crcerr_runtimecount"
		;;
	*)  # Default case
		echo "$1 not a supported uBoot environment vrbl."
		usage
		exit 1
		;;
	esac

#
# Verify that a positive integer is passed in
#
num=$2;
if [ "${num//[^0-9]/}" == "$num" ]; then
        if [ $2 -lt 0 ]
        then
                echo "Increment must be positive."
                usage
                exit 1
        fi
else
        echo "$2 is not a valid integer."
        usage
        exit 1
fi

# All inputs are valid.
#
# Search through existing uBoot environment variables.
# If environment variable already exists, add the increment
# to the existing value, and save it.
#
# If the environment variable does not yet exist,
# create it, and initialize it to the provided value.
#

# Examine existing uBoot environment variables.
/usr/local/bin/fw_printenv | awk -v env_vrbl=$1 -v in_count=$2                  \
        'BEGIN {value=in_count;};                                               \
        $1 ~ env_vrbl {                                                         \
                # environment variable was found in list                        \
                found=1;                                                        \
                # All valid env. variables will be in the form                  \
                # vrbl=value.  Extract the value field.                         \
                search="=";                                                     \
                n=split($1, temp_array, search);                                \
                if (n == 2) {                                                   \
                        # make sure existing value is valid integer.            \
                        value = temp_array[2];                                  \
                        if (value ~ /[0-9]+/                                    \
                            && value !~ /[a-zA-Z\.\-]+/) {                      \
                                # Value is valid, increment it                  \
                                value+=in_count;                                \
                        }                                                       \
                        else                                                    \
                        {                                                       \
                                # Value is not valid. Add a valid value to the  \
                                # environment by treating it as "not-found"     \
                                # and letting the END handle it.                \
                        }                                                       \
                }                                                               \
        }                                                                       \
        END {                                                                   \
                # Add variable to the uBoot environment                         \
                # An existing variable will be updated, or a new variable       \
                # will be created here.                                         \
                commandstring="/usr/local/bin/fw_setenv " env_vrbl " " value;   \
                system(commandstring);                                          \
        }'

exit 0

