#!/bin/sh

. /usr/local/bin/sig_catcher


UPDATE_BOOT=0
UPDATE_FIRMWARE=0

check_current_version() {
	#
	# get current version on mpc
	#

	installed_version=`/usr/local/bin/mp_controller $target version | awk '{print $2}'`
	prefix=`echo $installed_version | cut -c1`
	if [ $prefix != 'M' ]; then
	    echo "Error retrieving current version"
	    exit 1
	fi
	echo "Checking current mpc firmware version on $target... $installed_version"

	operating_mode=`echo $installed_version |cut -c2-2`
	#echo "mode: $operating_mode"

	minor_rev=`echo $installed_version | cut -c5-5`
	#echo "rev: $minor_rev"

	if [ $minor_rev -lt 1 ]; then
	    echo "Rev must be 1.1 or greater"
	    exit 1
	fi
}

check_if_in_boot_mode() {
	#
	# get current version on mpc
	#
	installed_version=`/usr/local/bin/mp_controller $target version | awk '{print $2}'`

	#echo "installed version: $installed_version"

	operating_mode=`echo $installed_version | cut -c2-2`
	#echo "mode: $operating_mode"

	if [ "$operating_mode" != "B" ]; then
	    # retry once
		installed_version=`/usr/local/bin/mp_controller $target version | awk '{print $2}'`
		operating_mode=`echo $installed_version |cut -c2-2`
		if [ "$operating_mode" != "B" ]; then
	    	echo "Must be in boot mode"
	    	exit 1
		fi
	fi
}

check_if_in_controller_mode() {
	#
	# get current version on mpc
	#
	installed_version=`/usr/local/bin/mp_controller $target version | awk '{print $2}'`
	#echo "installed version: $installed_version"

	operating_mode=`echo $installed_version |cut -c2-2`
	#echo "mode: $operating_mode"

	if [ "$operating_mode" != "C" ]; then
	    # retry once
		installed_version=`/usr/local/bin/mp_controller $target version | awk '{print $2}'`
		operating_mode=`echo $installed_version |cut -c2-2`
		if [ "$operating_mode" != "C" ]; then
	    	echo "Must be in controller mode"
	    	exit 1
		fi
	fi
}

find_upgrade_firmware() {
	#
	# find upgrade firmware in /firmware
	#

	new_firmware_name=`ls -L /firmware/Carnegie/switch |grep "marvell_prom_controller" | grep -v boot | sort | tail -n 1 `
	if [ "$new_firmware_name" == "" ]; then
	    echo "Can't find new firmware in /firmware/Carnegie/switch"
	    exit 1
	fi

	echo "Locating new firmware... $new_firmware_name"
	new_firmware_name="/firmware/Carnegie/switch/$new_firmware_name"
}

find_upgrade_boot() {
	#
	# find upgrade boot in /firmware
	#

	new_boot_name=`ls -L /firmware/Carnegie/switch |grep "marvell_prom_controller" | grep boot | sort | tail -n 1`
	if [ "$new_boot_name" == "" ]; then
	    echo "Can't find new boot in /firmware/Carnegie/switch"
	    exit 1
	fi

	echo "Locating new boot... $new_boot_name"
	new_boot_name="/firmware/Carnegie/switch/$new_boot_name"
}

switch_to_boot_mode() {

	echo "Switching to bootloader"
	/usr/local/bin/mp_controller $target jumpto_boot
	sleep 3 
}

switch_to_controller_mode() {

	echo "Switching to controller mode"
	/usr/local/bin/mp_controller $target jumpto_firmware
	sleep 3 
}

controller_reset() {

	echo "Resetting controller"
	/usr/local/bin/mp_controller $target reset
	sleep 3
}

erase_firmware() {

	echo "Erasing firmware..."
	/usr/local/bin/mp_controller $target erase_firmware

	if [ $? -eq 1 ]; then
	    echo "Failure erasing firmware"
	    exit 1
	fi
}

erase_boot() {

	echo "Erasing boot..."
	/usr/local/bin/mp_controller $target erase_boot

	if [ $? -eq 1 ]; then
	    echo "Failure erasing boot"
	    exit 1
	fi
}

program_firmware() {

	echo "Programming firmware..."
	/usr/local/bin/mp_controller $target program_firmware $new_firmware_name

	if [ $? -eq 1 ]; then
	    echo "Failure programming firmware"
	    exit 1
	fi
}

program_boot() {

	echo "Programming boot..."
	/usr/local/bin/mp_controller $target program_boot $new_boot_name

	if [ $? -eq 1 ]; then
	    echo "Failure programming boot"
	    exit 1
	fi
}


usage() {
    echo "Usage:  $0 [option] <target>"
    echo "           Options:"
    echo "           -a: All images (firmware and boot)"
    echo "           -b: Only the boot image"
    echo "           -f: Only the firmware image"
    echo ""
    echo "           <target> is  0 = mb, 1 = ioxm:"
    exit 1 
}

if ! [ -f /var/run/Carnegie ]; then
    echo "Command not applicable on this platform"
    exit 1
fi

if [ $# -lt 1 ]; then
    usage
else
	while getopts abf c 
	do
		case $c in
			a)
				UPDATE_BOOT=1
				UPDATE_FIRMWARE=1
				;;
			b)
				UPDATE_BOOT=1
				;;
			f)
				UPDATE_FIRMWARE=1
				;;
			*)
				usage
				;;	
		esac
		shift
	done
fi

if [ $UPDATE_BOOT == 0 ] && [ $UPDATE_FIRMWARE == 0 ]; then
    echo "No options given. Nothing to do."
    exit 1
fi

#
# first argument is target 0=mb, 1=ioxm
#
target=$1

if [ $target -lt 0 ] || [ $target -gt 1 ]; then
	usage
fi


if [ $UPDATE_FIRMWARE == 1 ]; then
	
	check_current_version

	find_upgrade_firmware
	
	switch_to_boot_mode

	check_if_in_boot_mode

	check_current_version

	erase_firmware

	program_firmware

	switch_to_controller_mode

#	controller_reset

fi

if [ $UPDATE_BOOT == 1 ]; then
	
	check_current_version

	find_upgrade_boot
	
	switch_to_controller_mode

	check_if_in_controller_mode

	check_current_version

	erase_boot

	program_boot

#	controller_reset
	

fi


echo "Done"
