#!/bin/ash
# This script reconfigures the RLM with the info
# from the $tuples.

####################################
#    FILES CHECKING
####################################

ifup="/sbin/ifup"
ifdown="/sbin/ifdown"


cmd_hostname="/bin/hostname"
do_firewall=0
fw_printenv="/usr/local/bin/fw_printenv"
tuples="/mnt/logs/etc/rlm_config_from_filer"
interfaces="/mnt/logs/etc/network/interfaces"
tmp_if="/tmp/interfaces"
route_info="/tmp/route_info"

hostname="/mnt/logs/etc/hostname"
tmp_dns="/tmp/config/etc/resolv.conf"
dns="/mnt/logs/etc/resolv.conf"

firewall="/etc/init.d/firewall"
IPv4="/sbin/ip"
log="/var/log/messages"
date=$(date | sed "s/[A-Z]\{3\} [0-9]\{4\}//")
tmp_ps="/tmp/rlm_ps"
IPTABLES="/sbin/iptables"
IP6TABLES="/sbin/ip6tables"
sp_ssh_conf="/mnt/logs/etc/sp_ssh_conf"
do_eth1=1
max_backup=3

rm -f $tmp_if

brdrev=$(cat /var/run/boardrev)
razorl_rev_p2i=3

if [ -f "/var/run/RazorL" ] && [ "$brdrev" -lt "$razorl_rev_p2i" ] ; then
    ETH0=eth0
else
    ETH0=eth0.32
fi

IFCONFIG="/sbin/ifconfig"

get_config_value() {

    search_for=$1
    if [ -z "$2" ] ; then
        what_file=${tuples}
    else
        what_file=${2}
    fi

    if [ -z "$search_for" ] || [ ! -s "$tuples" ] ; then
        echo ""
    else
        grep -v "^#" < ${what_file} | grep "^$search_for " | tail -1 | sed 's/\([^ \t]*\)\([ \t]*\)\(.*\)/\3/g' 2> /dev/null
    fi

}


create_base_section() {

    if gpiotool 0 10 --get-data | grep High &> /dev/null ; then
        slot=2
    else
        slot=1
    fi

    greenip=$((${slot}+2))

cat << ENDOFFILE
# this file is generated
auto lo
iface lo inet loopback

auto eth0
iface eth0 inet manual
    up ifconfig eth0 0.0.0.0 mtu 1400 up
    up vconfig add eth0 32
    up vconfig add eth0 16
    up vconfig add eth0 64
    down ifdown eth0.16
    down ifdown eth0.32
    down ifdown eth0.64
    down vconfig rem eth0.32
    down vconfig rem eth0.16
    down vconfig rem eth0.64
    down ifconfig eth0 down

auto eth0.16
iface eth0.16 inet manual
    up conf_adapter.sh -i eth0.16 -a up
    down conf_adapter.sh -i eth0.16 -a down
    vlan-raw-device eth0

auto eth0.64
iface eth0.64 inet manual
    up conf_adapter.sh -i eth0.64 -a up
    down conf_adapter.sh -i eth0.64 -a down
    vlan-raw-device eth0

# the rest of this file changes based on
# the user configuration of the service processor

ENDOFFILE

}


#
# create the interface file
#
create_if_file() {

local wrote_auto

    wrote_auto=0

    if [ -f "/var/run/RazorL" ] && [ "$brdrev" -lt "$razorl_rev_p2i" ] ; then
        #we confirm tuples exists before running create_if_file
        awk -f /usr/local/bin/rlm_conf_awk_rules \
            dhcp_enabled=$is_dhcp_enabled \
            ipv4_enabled=$is_ipv4_enabled \
            ipv6_enabled=$is_ipv6_enabled \
            ipv6_global_present=$is_ipv6_global_present \
            $tuples
    else
        # vlans running, they have a completely
        # different way of generating the interfaces file
        # eth0 and lo always start for vespa

        create_base_section

    # if we're just using ipv4, this is good enough
        if [ "$is_dhcp_enabled" -eq "1" ] ; then
cat << Eof
# configuration for ipv4 that uses dhcp, no ipv6
auto eth0.32
iface eth0.32 inet dhcp
    mtu 1400
    vlan-raw-device eth0
Eof
        wrote_auto=1
        else
            # this device is running ipv4, usint static address
            if [ "$is_ipv4_enabled" -eq "1" ] ; then
                ip4addr=$(get_config_value ip)
                ip4gateway=$(get_config_value gateway)
                ip4netmask=$(get_config_value netmask)
                wrote_auto=1
                echo -e "\n\nauto $ETH0\niface $ETH0 inet static"
                echo -e "\tvlan-raw-device eth0\n\tmtu 1400"
                if [ -n "$ip4addr" ] ; then
                    echo -e "\taddress $ip4addr"
                    [ -n "$ip4gateway" ] && echo -e "\tgateway $ip4gateway"
                    [ -n "$ip4netmask" ] && echo -e "\tnetmask $ip4netmask"
                fi
            fi
        fi

        # check for ipv6
        if [ "$is_ipv6_enabled" -eq "1" ] ; then
            ip6addr=$(get_config_value ip6_addr)
            ip6addrprefix=$(get_config_value ip6_prefix)
            ip6gateway=$(get_config_value ip6_gate)
            [ "$wrote_auto" -eq "0" ] && echo -e "\nauto $ETH0"

            # this is for cases where the user did not specify an ipv6 address
            # so we need to create one so the ipv6 adapter can start and get an ra_address
            # if accept_ra is set.  Otherwise, this adatper will just have a link local address
            # with the busybox tools, I cannot start an ipv6 adatper without an address
            if [ -n "$ip6addr" ] ; then
                echo -e "\niface $ETH0 inet6 static"
                echo -e "\tvlan-raw-device eth0\n\tmtu 1400"
                echo -e "\taddress $ip6addr"
                [ -n "$ip6addrprefix" ] && echo -e "\tnetmask $ip6addrprefix"
                [ -n "$ip6gateway" ] && echo -e "\tgateway $ip6gateway"
            else
                echo -e "\n#$ETH0 configured without static ipv6 address"
                if  [ "$is_ipv4_enabled" -eq "0" ] ; then
                    echo -e "\n#and no ipv4, so start adapter manually"
                    echo -e "\niface $ETH0 inet manual"
                    echo -e "\n\tup ifconfig $ETH0 up"
                    echo -e "\n\tdown ifconfig $ETH0 down"
                fi
            fi
        fi
    fi
}

####################################
#    UP-CONVERT DNS CONFIG FILE
# This section (and only this) should be run if
# called during init i.e., with
# "dns_config" as command line parameter
# Normal invocations without any additional
# parameters will skip this section.
####################################
if [ $# -eq 1 ] && [ $1 = "dns_config" ]; then
   if [ -e $dns ] && [ -s $tuples ]; then
      tmp_dns1="/tmp/resolv.conf"
      grep -q "sp-auto-generated" $dns
      reconfig_needed=$?
      if [ "$reconfig_needed" -eq 1 ]; then
          dns_domainname=$(sed -n "s/^dns.domainname //p" $tuples)

         if [ "$dns_domainname" != "" ]; then
            echo "# sp-auto-generated by rlm_conf.sh at init time" > $tmp_dns1
            echo "search $dns_domainname" >> $tmp_dns1
            cat $dns >> $tmp_dns1
            mv -f $tmp_dns1 $dns
         fi

      fi
   fi
exit 0
fi

echo "1" > "/proc/sys/net/ipv4/conf/default/rp_filter"
echo "0" > "/proc/sys/net/ipv4/conf/default/arp_ignore"
echo "1" > "/proc/sys/net/ipv6/conf/default/accept_dad"

# BURT 945890 HACK. Same change as BURT 932283 (2015-Aug-06): Disable rp_filter to get eth0.64
# vlan to work again after kernel upgrade in 5.0X13002
#echo "1" > "/proc/sys/net/ipv4/conf/all/rp_filter"
echo "0" > "/proc/sys/net/ipv4/conf/all/rp_filter"

echo "0" > "/proc/sys/net/ipv4/conf/all/arp_ignore"
echo "1" > "/proc/sys/net/ipv6/conf/all/accept_dad"

#############################################
#  Generate process id (configd)
#############################################

ps > $tmp_ps
pid=$(awk '/configd/ {print $1}' $tmp_ps)
loginfo="$date (none) configd[$pid]:"
rm -f $tmp_ps
err=0
dhcp=0
is_v6_global_present=0

if [ -s $tuples ]; then
    chmod 0666 $tuples

    #############################################
    #    CHECK IF FIREWALL NEEDS TO BE RESTARTED
    #############################################

    is_dhcp_enabled=$(awk '/^dhcp/ {if ($2 == 1) print "1"; else print "0"}' $tuples)
    is_ipv4_enabled=$(awk '/^ipv4_disable/ {if ($2 == 1) print "0"; else print "1"}' $tuples)
    is_ipv6_enabled=$(awk '/^ipv6_disable/ {if ($2 == 1) print "0"; else print "1"}' $tuples)
    is_ipv6_global_present=$(awk '/^ip6_addr/ {print $2}' $tuples)
    is_ipv6_ra_disabled=$(awk '/^ra_disable/ {print $2}' $tuples)
else
    # If there is no tuples file, default to DHCP since we have
    # clearly don't have a static config
    is_dhcp_enabled=1
fi

if [ -z $is_ipv4_enabled ]
then
    is_ipv4_enabled=1
fi

if [ -z $is_ipv6_enabled ]
then
    is_ipv6_enabled=0
fi

if [ -z $is_dhcp_enabled ]
then
    is_dhcp_enabled=0
fi

if [ -z $is_ipv6_global_present ]
then
    is_ipv6_global_present=0
else
    is_ipv6_global_present=1
fi

if [ -z $is_ipv6_ra_disabled ]
then
    is_ipv6_ra_disabled=0
fi

if [ -s $tuples ]
then
    force_adapter_config=0
    # read the tuples files and create the interface ($tmp_if) file
    create_if_file > $tmp_if

    is_v6_global_present=$(grep -c -m 1 "inet6 static" $tmp_if)

    if [ "$is_ipv6_enabled" -eq 0 ]
    then
        echo "#ipv6 disabled" >> $tmp_if
        [ "$(ip -6 -o addr show | sed 's/  */ /g' | cut -d" " -f 4 | wc -l)" -gt 0 ] && force_adapter_config=1
    else
        force_adapter_config=1
        if [ "$is_v6_global_present" -eq 0 ]
        then
            echo "#ipv6 enabled without Global address" >> $tmp_if
        fi
        if [ "$is_ipv6_ra_disabled" -eq 1 ]
        then
            echo "#ipv6 enabled without Router Assigned address" >> $tmp_if
        fi
        # fetch the gateway from the configuration file
        ipv6_default_gateway=$(grep ^ip6_gate < ${tuples} | cut -d' ' -f 2)
    fi

    #compute the diff
    diff $interfaces $tmp_if >> $log
    return_diff=$?
    echo return_diff $return_diff >> $log

    if [ $return_diff != 0 ] || [ "$force_adapter_config" -eq 1 ]; then
        do_firewall=1
    fi

    # if the user as disabled dhcp, remove the dhcp address file
    # as this is used to determine if the IP address has changed between
    # lease expirations.
    if [ "$is_dhcp_enabled" -ne "1" ] ; then
        rm -f /var/dhcp/ip
    fi
fi


##############################################################
#    CHECK IF BOTH INTERFACES AND TUPLES FILES ARE NOT PRESENT
##############################################################

if [ ! -s "$interfaces" ] && [ ! -s "$tuples" ] ; then
    if [ -f "/var/run/RazorL" ] && [ "$brdrev" -lt "$razorl_rev_p2i" ] ; then
        is_dhcp_enabled=0
        echo "# This file is generated by RLM upon request from Ontap." > tmp_if
        echo "# Do NOT modify this file manually." >> tmp_if
        echo "auto eth0" >> tmp_if
        echo "iface eth0 inet static" >> tmp_if
        echo "   address 192.168.220.2" >> tmp_if
        echo "   netmask 255.255.255.0" >> tmp_if
        echo "   gateway 192.168.220.1" >> tmp_if
        echo "#ipv6 disabled" >> $tmp_if
    else
        create_base_section > $tmpif
    fi

    do_firewall=1
fi

####################################################
#    CHECK IF GLOBAL V6 ADDRESS NEEDS TO BE DELETED
####################################################

if [ "$is_ipv6_enabled" -eq 1 ]
then
	echo "1" > "/proc/sys/net/ipv6/conf/$ETH0/accept_dad"
	echo "0" > "/proc/sys/net/ipv6/conf/$ETH0/disable_ipv6"

    if [ -s "$tuples" ] ; then
        new_ip6_addr=$(awk '/^ip6_addr/ {print $2}' $tuples)
    fi

    inet_val=$(awk "/^$ETH0 inet / {print $4}" $interfaces)

    if [ -z "${inet_val}" -o "${inet_val}" = "dhcp" ]
    then
        # interfaces file is going to have just 1 line for
        # each of address and netmask fields
        cur_ip6_addr=$(awk '/^address / {print $2}' $interfaces)
        cur_net6=$(awk '/^netmask / {print $2}' $interfaces)
    else
        # interfaces file is going to have 2 lines for
        # each of address and netmask fields (v4 and v6)
        # pick the second line in this case
        cur_ip6_addr=$(awk '/^address / {if (i==1) print $2; i++;}' $interfaces)
        cur_net6=$(awk '/^netmask / {if (i==1) print $2; i++;}' $interfaces)
    fi

    if [ -z "${new_ip6_addr}" ]
    then
        if [ -n "$cur_ip6_addr" ]
        then
            $($IFCONFIG $ETH0 del "$cur_ip6_addr"/"$cur_net6")
        fi
    fi
fi

###########################################################################
#    sp_ssh_config tuple is not present in ONTAP version < IC.6 and also in
#       versions that doesn't support RLM/SP SSH access control feature
#    in that case flush ssh_pkts chain in the firewall
############################################################################
if [ -s "$tuples" ] ; then
    is_ssh_access_enabled=$(grep -c -m 1 sp_ssh_config $tuples)
fi

if [ -n "$is_ssh_access_enabled" ] ; then
    if [ "$is_ssh_access_enabled" -eq 0 ]
    then
        rm -f $sp_ssh_conf
        $IPTABLES -F ssh_pkts

        #Following 2 rules create a 'recent' list of recently seen IP addresses.
        #If new SSH connection requests are made from the same IP in a short
        #interval, it is likely a DOS attack. The following rule allows 10 SSH
        #requests from the same IP within a minute and then drops new requests.

        $IPTABLES -A ssh_pkts -p TCP -i $ETH0 --dport 22 -m state \
            --state NEW -m recent --set
        $IPTABLES -A ssh_pkts -p TCP -i $ETH0 --dport 22 -m state \
            --state NEW -m recent --update --seconds 60 --hitcount 10 -j DROP

        #Following rule rate-limits new incoming ssh connection requests to
        #an average of 20 per minute, with an initial burst of 20 connections
        $IPTABLES -A ssh_pkts -p TCP -i $ETH0 -s 0/0 --dport 22 --syn \
            -m state --state NEW -m limit --limit 20/minute --limit-burst 20 -j ACCEPT

        $IP6TABLES -F ssh_pkts
        $IP6TABLES -A ssh_pkts -p tcp -i $ETH0 -s 0/0 --dport 22 -j ACCEPT
    fi
fi

####################################
#    DNS RESOLUTION
####################################
# dns is optional so do not check the copy
if [ -e "$tmp_dns" ] && [ -s "$tuples" ]; then
    dns_domainname=$(sed -n "s/^dns.domainname //p" $tuples)
   if [ "$dns_domainname" != "" ]; then
      echo "# sp-auto-generated by rlm_conf.sh invoked by configd" > $dns
      echo "search $dns_domainname" >> $dns
      cat $tmp_dns >> $dns
   else
      cp $tmp_dns $dns
   fi
fi


####################################
#    IP TABLES
####################################
# Rewrite the file anyway to get the correct format
if [ -e "$tmp_if" ]
then
    # save a backup of the previous interfaces file (only if there are changes)
    diff -w -B $tmp_if $interfaces &> /dev/null
    if [ $? = 1 ]
    then
        cp $interfaces $interfaces.$(date +%Y%m%d%H%M%S)
    fi

    # limit the number of backups we save to the newest 3 files
    num_prev_interfaces=$(ls -la /mnt/logs/etc/network/interfaces.*[0-9] | wc -l)
    if [ $num_prev_interfaces -gt $max_backup ]
    then
        #list all timestamped interfaces file, newest first, and remove all but the newest 3
        ls -c $interfaces.*[0-9] | tail -n $((num_prev_interfaces - max_backup)) | xargs rm -f
    fi

    cp $tmp_if $interfaces || {
       echo "$loginfo can't save $interfaces" 1>>$log
       err=34
       exit $err
    }
    sync
fi

/usr/local/bin/net_link_up.sh;
if ! [ $? = 1 ]; then {
   echo "# no link"  >> $interfaces
   err=28
   exit $err
}
fi

is_v4_static_present=$(cat /mnt/logs/etc/interfaces | grep -c -m 1 "inet static")
is_eth0_present=$(cat /mnt/logs/etc/interfaces | grep -c -m 1 $ETH0)
is_eth1_present=$(cat /mnt/logs/etc/interfaces | grep -c -m 1 eth1)
if [ "$do_firewall" -eq 1 ]; then

        # Before running the firewall, the eth0 interface needs to be
        # brought down and then up.
        #
        # There are 4 statements below that do this job
        # 1. ifdown eth0
        # 2. ifconfig eth0 down
        # 3. ifup eth0
        # 4. ifconfig eth0 up
        #
        # 1 & 3 are always run
        # 2 & 4 are optional and are run during just this case,
        # Whenever there
        #   - IPv6 is enabled
        #   - No v6 entry in the /mnt/logs/etc/interfaces file
        #   - DHCPv4 or no IPv4 entry is present in /mnt/logs/etc/interfaces
        #
        # Reasons:
        #
        # 1. ifdown eth0 and ifup eth0
        #   --------------------------
        #   Will only work on addresses present in the /etc/network/interfaces
        #   --------------------------
        #   Will only work on addresses present in the /etc/network/interfaces
        #   --------------------------
        #   Will only work on addresses present in the /etc/network/interfaces
        #   --------------------------
        #   Will only work on addresses present in the /etc/network/interfaces
        #   file. And since LLA and RA is brought up by the kernel
        #   and this info is not present in the interfaces file, we
        #   need the below 2 statements to bring up RA and LLA of RLM
        #
        # 2. ifconfig eth0 down and ifconfig eth0 up
        #   ----------------------------------------
        #   These 2 have no effect on IPv4 address but will bring back IPv6
        #   addresses which are native to the kernel and not present in the
        #   /etc/network/interfaces file
        #
        # 3. Same Ordering of these statments are important as well.

        # NOTE: Kernel brings up v6 interfaces only in 2 cases:
        # 1. When interfaces file have Global v6 Address, through "ifup eth0"
        # 2. When interfaces file does not have any v6 entry, but v6 is enabled
        # on RLM, then "ifconfig eth0 up" will bring up the native IPv6
        # addresses back to life.
        # Burt reference 361431

    $ifdown $ETH0 >> $log
    ifdown_status=$?
    echo "ifdown status $ifdown_status" >> $log

    if [ "$ifdown_status" -eq 1 ] && [ "$is_eth0_present" -eq 1 ]; then
        # Do not abort $ETH0 configuration prcess without attempting to recover.
        # Refer burt 1192879 for additional details.
        #
        # If there is a UDHCPC client running before, kill it completely
        # Refer burt 344656 for more information
        if [ -e "/var/run/udhcpc.$ETH0.pid" ]
        then
            kill $(cat /var/run/udhcpc.$ETH0.pid) 2>/dev/null
        fi
        RETRY=0
        MAX_RETRY=3
        while [ $RETRY -lt $MAX_RETRY ] ; do
            echo "Retrying $ifdown -f $ETH0" >> $log
            sleep 1
            $ifdown -f $ETH0 >> $log
            ifdown_status=$?
            echo "ifdown status $ifdown_status" >> $log
            if [ "$ifdown_status" -eq 1 ] && [ "$is_eth0_present" -eq 1 ]; then
                RETRY=`expr $RETRY + 1`
            else
                break
            fi
        done
        # Do not abort $ETH0 config process if 'ifdown -f $ETH0' fails, but
        # there is no inet addr associated with $ETH0.
        if [ $RETRY -ge $MAX_RETRY ] ; then
            $IFCONFIG $ETH0 | grep 'inet addr' > /dev/null
            ifconfig_status=$?
            if [ $ifconfig_status -eq 0 ] ; then
               echo "$loginfo ifdown failed to bring down $ETH0" 1>>$log
               err=36
               exit $err
            fi
        fi
    fi

    if [ "$is_ipv6_enabled" -eq 1 ] && [ "$is_v6_global_present" -eq 0 ] && [ "$is_v4_static_present" -eq 0 ]; then
        $IFCONFIG $ETH0 down
        if [ $? -eq 1 ]; then
            echo "$loginfo ifconfig failed to bring down $ETH0" 1>>$log
            err=38
            exit $err
        fi
    fi

    # If there is a UDHCPC client running before, kill it completely
    # Refer burt 344656 for more information
    if [ -e "/var/run/udhcpc.$ETH0.pid" ]; then
        kill $(cat /var/run/udhcpc.$ETH0.pid) 2>/dev/null
    fi

    if [ "$is_ipv6_enabled" -eq 0 ] ; then
        [  -e "/proc/sys/net/ipv6/conf/$ETH0/accept_ra" ] && (echo "0" > "/proc/sys/net/ipv6/conf/$ETH0/accept_ra")
	else
		echo "1" > "/proc/sys/net/ipv6/conf/$ETH0/accept_dad"
		echo "0" > "/proc/sys/net/ipv6/conf/$ETH0/disable_ipv6"
    fi

	echo "1" > "/proc/sys/net/ipv4/conf/eth0/rp_filter"
	echo "1" > "/proc/sys/net/ipv4/conf/eth0/arp_ignore"

    # the adpaters aren't properly configured, so we need to manually add in the loopback address
    ip addr show | grep "inet6 ::1" &> /dev/null || ip addr add ::1 dev lo
    $ifup $ETH0 >> $log
    ifupstatus=$?
    echo "ifup status $ifupstatus" >> $log
    if [ "$ifupstatus" -eq 1 ] && [ "$is_eth0_present" -eq 1 ]; then
        echo "$loginfo ifup failed to bring up $ETH0" 1>>$log
    fi


    if [ "$is_ipv6_enabled" -eq 1 ] && [ "$is_v6_global_present" -eq 0 ] && [ "$is_v4_static_present" -eq 0 ]; then
        $IFCONFIG $ETH0 up
        if [ $? -eq 1 ]; then
            echo "$loginfo ifconfig failed to bring up $ETH0" 1>>$log
            err=39
            exit $err
        fi

        # Set this value if the RA needs to be disabled.
        # Even though this will not delete the RA IPv6 address present in ifconfig,
        # RLM will no longer accpet any new RA addresses after this
        # Deletion of the RA will be done in the firewall
        if [ "$is_ipv6_enabled" -eq 1 ] ; then
            # adjust the MTU to 1400 bytes, this makes some routers Happy at the sake of performance.
            # burt: 427425
            ${IFCONFIG} $ETH0 mtu 1400 2>&1 >> $log
            for d in 16 64 ; do
                [ -f "/proc/sys/net/ipv6/conf/eth0.$d/accept_ra" ] && (echo "0" >  "/proc/sys/net/ipv6/conf/eth0.$d/accept_ra")
            done
            if [ -e "/proc/sys/net/ipv6/conf/$ETH0/accept_ra" ] ; then
			if [ "$is_ipv6_ra_disabled" -eq 1 ] ; then
			echo "0" > "/proc/sys/net/ipv6/conf/$ETH0/accept_ra"
                else
                    echo "1" > "/proc/sys/net/ipv6/conf/$ETH0/accept_ra"
                fi
            else
                echo "rlm_conf.sh - /proc/sys/net/ipv6/conf/$ETH0/accept_ra, file not found" >> $log
            fi
        fi


        # ifconfig eth0 down and ifconfig eth0 up commands remove the
        # default entry for route.  Following code adds them back.
        if [ "$is_ipv4_enabled" -eq 1 ] ; then
        # not sure how route_info ever gets populated
            gateway=$(awk '/^gateway/ {print $2}' $route_info)
            interface=$(awk '/^interface/ {print $2}' $route_info)
        else
            if [ -s "$tuples" ] ; then
                gateway=$(awk '/^gateway/ {print $2}' $tuples)
            else
                echo "No tuples file (rlm_config_from_filer).  Not setting default gw"
            fi
            interface=$(awk '/^inet static/ {print $2}' $interfaces)
        fi

        if [ -n "$gateway" ] ; then
            route add default gw $gateway $interface
        fi

        if [ "$is_ipv6_enabled" -eq 1 ] ; then
            if [ -s "$tuples" ] ; then
                ipv6_gateway=$(grep ^ip6_gate $tuples | cut -d " " -f 2)
            fi
            [ -n "$ipv6_gateway" ] && ip -6 route add default via "$ipv6_gateway" >> $log
        fi

    fi

    if [ "$do_eth1" -eq 1 ] ; then
        $ifup eth1 >> $log
        ifupstatus=$?
        echo "ifup status $ifupstatus" >> $log
        if [ "$ifupstatus" -eq 1 ] && [ "$is_eth1_present" -eq 1 ]; then
            echo "$loginfo ifup failed to bring up eth1" 1>>$log
        fi

        $IFCONFIG eth1 up
        if [ $? -eq 1 ]; then
            echo "$loginfo ifconfig failed to bring up eth1" 1>>$log
            err=39
            exit $err
        fi
    fi

    if [ "$is_ipv6_enabled" -eq 0 ] || [ "$is_ipv6_ra_disabled" -eq 1 ] ; then
        # belt and suspenders, make sure that accept_ra isn't enabled when ipv6 isn't enabled.
        for d in 16 32 64 ; do
            [  -e "/proc/sys/net/ipv6/conf/eth0.$d/accept_ra" ] && (echo "0" > "/proc/sys/net/ipv6/conf/eth0.$d/accept_ra")
        done
        [  -e "/proc/sys/net/ipv6/conf/eth0/accept_ra" ] && (echo "0" > "/proc/sys/net/ipv6/conf/eth0/accept_ra")
    fi

    # remove left-over loop back adatper address for IPv6
    # if we're not using ipv6
    # burt: 425245
    if [ "$is_ipv6_enabled" -eq 0 ] ; then

        # we have straggler addresses left in the adapter when
        # ipv6 has been enabled, disabled, the re-enabled
        # remove those now
        ip -6 -o addr show | sed 's/  */ /g' | cut -d" " -f 4,2 | while read d a ; do
            [ -n "$a" ] && ip addr del $a dev $d
        done
        # make extra-double sure that the loop back adpater is removed (above loop should remove it,
        # but the network code is such patchwork that this is necessary nonetheless)
        ip addr show | grep "inet6 ::1" &> /dev/null && ip addr del ::1 dev lo
    else

        if [ "$is_ipv6_ra_disabled" -eq 1 ] ; then
            # the default RA address may still be present, remove that
            old_ra_addr=$(ip -6 route show default | grep -v ${ipv6_default_gateway} | grep -v ^unreachable | cut -d" " -f 3)
            if [ -n "$old_ra_addr" ] ; then
                ip -6 route delete default via $old_ra_addr
            fi

            # likewise, the default RA address may be there too
            old_ra_addr=$(ip -o -6 addr show dev $ETH0 scope global | grep dynamic | awk '{print $4}'| cut -d/ -f1)
            if [ -n "$old_ra_addr" ] ; then
                ip -6 addr del ${old_ra_addr} dev $ETH0
            fi
        else
            [ -e "/proc/sys/net/ipv6/conf/$ETH0/accept_ra" ] && echo 1 > /proc/sys/net/ipv6/conf/$ETH0/accept_ra
        fi

        # burt 436305
        # thump the device over head to add the default gateway when not set correctly
        if ! ip -6 route show | grep ^default | grep ${ipv6_default_gateway} &> /dev/null ; then
            echo "cannot find default gateway ${ipv6_default_gateway}, setting manually" | logger
            ip -6 route show | logger
            ip -6 route delete default 2>&1 | logger
            ip -6 route add default via ${ipv6_default_gateway} dev $ETH0 2>&1 | logger
        fi

        # if the device was re-configured, we need to adjust the
    fi

    $firewall restart || {
        echo "$loginfo firewall starter failed, err: $ret" 1>>$log
        err=43
        exit $err
    }

    if [ "$is_ipv4_enabled" -eq "1" ] ; then
       # Added test for DHCP not enabled, since if DHCP is enabled, the IP tag in rlm_conf_from_filer is blank
       # and the arping command is invalid.
       # Also added sending the arping results to logger, since if arping doesn't return a response, the return
       # value of the script is non-zero, which is interpreted as a failure by the caller, and it causes
       # sp setup to fail.
       if [ "$is_dhcp_enabled" -ne "1" ] ; then
          arping -c2 -I $ETH0 $(get_config_value ip)  2&>1 | logger
       fi
    fi

	ip6addr=$(grep ip6_addr < /mnt/logs/etc/rlm_config_from_filer | cut -d ' ' -f 2)
    if [ "$is_ipv6_enabled" -eq "1" -a -n "$ip6addr" ] ; then
            /usr/local/bin/ipv6_nd.sh &
    fi
fi

