#!/bin/ash
# This script gathers the RLM configuration info to a tuple file
# and put it in $tuples_to_filer for filer's use.

set -x
fw_printenv="/usr/local/bin/fw_printenv"

ftpdir="/tmp/config"
final_tuples="$ftpdir/rlm_config_to_filer"
tuples_to_filer="/mnt/logs/etc/rlm_config_to_filer"
tuples_xtd="/mnt/logs/etc/rlm_config_xtd_info"

tuples_from_filer="/mnt/logs/etc/rlm_config_from_filer"
stats="$ftpdir/rlm_stats"
interfaces="/etc/network/interfaces"
tmp_ifconfig_eth0="/tmp/ifconfig_file_eth0"
tmp_ip_6_addr="/tmp/ip_6_addr"

log="/var/log/messages"
date=$(date | sed "s/[A-Z]\{3\} [0-9]\{4\}//")
tmp_ps="/tmp/rlm_ps"
ps > $tmp_ps
pid=$(awk '/configd/ {print $1}' $tmp_ps)
loginfo="$date (none) configd[$pid]:"
rm -f $tmp_ps

err=0

brdrev=$(cat /var/run/boardrev)
razorl_rev_p2i=3

if [ -f "/var/run/RazorL" ] && [ "$brdrev" -lt "$razorl_rev_p2i" ] ; then
    ETH0=eth0
else
    ETH0=eth0.32
fi

####################################
#    FILES CHECKING
####################################

rm -f $tuples_to_filer
rm -f $tuples_xtd
rm -f $tmp_ifconfig_eth0
rm -f $tmp_ip_6_addr

sleep 3
ifconfig ${ETH0} > $tmp_ifconfig_eth0
/sbin/ip -6 addr show dev ${ETH0} > $tmp_ip_6_addr


if [ -d "$ftpdir" ]
then
    : # no-op
else
    mkdir $ftpdir || {
    echo "$loginfo mkdir $ftpdir failed" 1>>$log
    exit 34
}
fi


########################################
#  Fru Info, Software version, statistic
########################################
#$fw_printenv | sed -n "
#/serial#/s/#=/ /p
#/part#/s/#=/ /p
#/rev#/s/#=/ /p" >>$tuples_to_filer || {
#    echo "$loginfo fail to get env info" 1>>$log
#    err=25
#}

echo -n "version " >>$tuples_to_filer
/usr/local/bin/version.sh -e -r >>$tuples_to_filer

/usr/local/bin/rlm_stats -p > $stats || {
    echo "$loginfo fail to get stats" 1>>$log
    err=34
}

#############################################
#  get v4, v6 enabled status
#############################################
if [ -s "$tuples_from_filer" ] ; then
    is_v4_disabled=$(awk '/ipv4_disable/ {print $2}' $tuples_from_filer)
    is_v6_disabled=$(awk '/ipv6_disable/ {print $2}' $tuples_from_filer)
    is_ipv6_global_present=$(awk '/ip6_addr/ {print $2}' $tuples_from_filer)
fi

if [ -z "$is_v4_disabled" ]
then
    is_v4_disabled=0
fi

if [ -z "$is_v6_disabled" ]
then
    is_v6_disabled=1
fi

if [ -z "$is_ipv6_global_present" ]
then
    is_ipv6_global_present=0
else
    is_ipv6_global_present=1
fi

if [ -z "$is_dhcp_enabled" ]
then
    is_dhcp_enabled=0
fi

if [ -e "/var/run/udhcpc.${ETH0}.pid" ] ; then
    is_dhcp_enabled=1
else
    is_dhcp_enabled=0
fi

##################################################################
# Update "rlm_config_to_filer" with the following information
#    HW Address
#    dhcp
#    ip
#    netmask
#    gateway
##################################################################

awk -v tuples_tofiler=$tuples_to_filer \
     '/HWaddr/ {print "MAC "$5 >> tuples_tofiler}' \
     $tmp_ifconfig_eth0

# update dhcp information
if [ "$is_dhcp_enabled" -eq 1 ]
then
    echo "dhcp 1" >> $tuples_to_filer
else
    echo "dhcp 0" >> $tuples_to_filer
fi

# if there is no cable connected, set the values as unknown
/usr/local/bin/net_link_up.sh;
if ! [ $? = 1 ]; then {

    echo "$loginfo no ethernet link" 1>>$log

    ip_params=$(awk '/inet addr/ {split($2, a, ":"); print a[2]}' $tmp_ifconfig_eth0)
    if [ "${#ip_params}" -gt 0 ]; then
        awk -v tuples_tofiler=$tuples_to_filer \
        '/inet addr/ {split($2, a, ":"); print "ip " a[2] >> tuples_tofiler} \
        /Mask/ {split($4, a, ":"); print "netmask " a[2] >> tuples_tofiler}' \
        $tmp_ifconfig_eth0
        route -n | awk -v tuples_tofiler=$tuples_to_filer \
        '{if ($1 == "0.0.0.0") print "gateway " $2 >> tuples_tofiler}'
    else
        echo "ip unknown" >> $tuples_to_filer
        echo "netmask unknown" >> $tuples_to_filer
        echo "gateway unknown" >> $tuples_to_filer
    fi

    echo "link_ipv6 0" > $tuples_xtd
    echo "link_ipv6_prefix 0" >> $tuples_xtd
    echo "link_ipv6_valid 0" >> $tuples_xtd
    echo "router_ipv6 0" >> $tuples_xtd
    echo "router_ipv6_prefix 0" >> $tuples_xtd
    echo "router_ipv6_valid 0" >> $tuples_xtd
    echo "global_ipv6 0" >> $tuples_xtd
    echo "global_ipv6_prefix 0" >> $tuples_xtd
    echo "global_ipv6_valid 0" >> $tuples_xtd
    echo "gateway_ipv6 0" >> $tuples_xtd

    cp $tuples_to_filer $final_tuples || {
        echo "$loginfo fail to save $final_tuples" 1>>$log
        err=34
    }

    rm -f $tmp_ifconfig_eth0
    rm -f $tmp_ip_6_addr

    exit $err
}
fi

# extract inet addr, netmask from "ifconfig eth0" output
# get the gateway information from route command
if [ ${is_v4_disabled:-0} -eq 1 ]
then
    echo "ip Disabled" >> $tuples_to_filer
    echo "netmask Disabled" >> $tuples_to_filer
    echo "gateway Disabled" >> $tuples_to_filer
else
    awk -v tuples_tofiler=$tuples_to_filer \
     '/inet addr/ {split($2, a, ":"); print "ip " a[2] >> tuples_tofiler} \
     /Mask/ {split($4, a, ":"); print "netmask " a[2] >> tuples_tofiler}' \
     $tmp_ifconfig_eth0

    route -n | awk -v tuples_tofiler=$tuples_to_filer \
         '{if ($1 == "0.0.0.0") print "gateway " $2 >> tuples_tofiler}'
fi


# store v4 addresses as unknown if no v4 addresses are found
is_inet_addr_present=$(grep -c "^ip" $tuples_to_filer)
is_netmask_present=$(grep -c "^netmask" $tuples_to_filer)
is_gateway_present=$(grep -c "^gateway" $tuples_to_filer)

if [ ${is_inet_addr_present:-0} -eq 0 ]
then
    echo "ip unknown" >> $tuples_to_filer
fi

if [ ${is_netmask_present:-0} -eq 0 ]
then
    echo "netmask unknown" >> $tuples_to_filer
fi

if [ ${is_gateway_present:-0} -eq 0 ]
then
    echo "gateway unknown" >> $tuples_to_filer
fi

##################################################################
# Update "rlm_config_xtd_info" with the following information
#    v6 gateway
#    v6 link local
#    v6 global address, v6 global prefix
#    v6 router address, v6 router prefix
#    v6 link local address
#    v6 gateway address
##################################################################

# extract "v6 gateway" from interfaces file
awk -v tuples_xtd_file=$tuples_xtd \
     '/gateway/ {ret=index($2, ":"); \
                if (ret!=0) print "gateway_ipv6 "$2 >> tuples_xtd_file}' \
                $interfaces

awk -v tuples_xtd_file=$tuples_xtd \
   '/scope link/ {split($2, a, "/"); print "link_ipv6 " a[1] \
      >> tuples_xtd_file;print "link_ipv6_prefix " a[2] >> tuples_xtd_file} \
   /scope global dynamic $/ {split($2, a, "/"); print "router_ipv6 " a[1] \
      >> tuples_xtd_file;print "router_ipv6_prefix " a[2] >> tuples_xtd_file} \
   /scope global $|scope global tentative / {split($2, a, "/"); print "global_ipv6 " a[1] \
      >> tuples_xtd_file;print "global_ipv6_prefix " a[2] >> tuples_xtd_file}' \
   $tmp_ip_6_addr


if [ -e "$tuples_xtd" ]
then
    is_link_addr_present=$(grep -c "link_ipv6 " $tuples_xtd)
    if [ "$is_link_addr_present" -eq 0 ]
    then
        echo "link_ipv6 0" >> $tuples_xtd
        echo "link_ipv6_prefix 0" >> $tuples_xtd
        echo "link_ipv6_valid 0" >> $tuples_xtd
    else
        echo "link_ipv6_valid 1" >> $tuples_xtd
    fi

    is_router_ipv6_present=$(grep -c "router_ipv6 " $tuples_xtd)
    if [ "$is_router_ipv6_present" -eq 0 ]
    then
        echo "router_ipv6 0" >> $tuples_xtd
        echo "router_ipv6_prefix 0" >> $tuples_xtd
        echo "router_ipv6_valid 0" >> $tuples_xtd
    else
        echo "router_ipv6_valid 1" >> $tuples_xtd
    fi

    is_global_ipv6_present=$(grep -c "global_ipv6 " $tuples_xtd)
    if [ "$is_global_ipv6_present" -eq 0 ]
    then
        echo "global_ipv6 0" >> $tuples_xtd
        echo "global_ipv6_valid 0" >> $tuples_xtd
        echo "global_ipv6_prefix 0" >> $tuples_xtd
    else
        echo "global_ipv6_valid 1" >> $tuples_xtd
    fi

    is_gateway_ipv6_present=$(grep -c "gateway_ipv6 " $tuples_xtd)
    if [ "$is_gateway_ipv6_present" -eq 0 ]
    then
        echo "gateway_ipv6 0" >> $tuples_xtd
    fi
fi


# if v6 is enabled and no v6 addresses found, then store v6
# addresses as zero.  This could happen when v6 is enabled with no global
# address and RLM cannot get a router assigned address
if [ ${is_v6_disabled:-0} -eq 0 -a ${is_ipv6_global_present:-0} -eq 0 -a ! -e "$tuples_xtd" ]
then
    echo "link_ipv6 0" >> $tuples_xtd
    echo "link_ipv6_prefix 0" >> $tuples_xtd
    echo "link_ipv6_valid 0" >> $tuples_xtd
    echo "router_ipv6 0" >> $tuples_xtd
    echo "router_ipv6_prefix 0" >> $tuples_xtd
    echo "router_ipv6_valid 0" >> $tuples_xtd
    echo "global_ipv6 0" >> $tuples_xtd
    echo "global_ipv6_prefix 0" >> $tuples_xtd
    echo "global_ipv6_valid 0" >> $tuples_xtd
    echo "gateway_ipv6 0" >> $tuples_xtd
fi


##################################################################
# check for error conditions
##################################################################

is_hwaddr_present=$(grep -c "MAC" $tuples_to_filer);
if [ ${is_hwaddr_present:-0} -eq 0 ]
then
    echo "$loginfo fail to get MAC" 1>>$log
    err=44
fi

if [ ${is_v4_disabled:-0} -eq 0 ]
then
    is_dhcp_present=$(grep -c "dhcp" $tuples_to_filer)
    is_inet_addr_present=$(grep -c "ip " $tuples_to_filer)
    is_netmask_present=$(grep -c "netmask" $tuples_to_filer)
    is_gateway_present=$(grep -c "gateway" $tuples_to_filer)
    if [ "$is_dhcp_present" -eq 0 -o "$is_inet_addr_present" -eq 0 -o \
        "$is_netmask_present" -eq 0 -o "$is_gateway_present" -eq 0 ]
    then
        echo "$loginfo fail to get ipv4 network info" 1>>$log
    fi
fi

if [ ${is_v6_disabled:-0} -eq 0 ]
then
    is_link_addr_present=$(grep -c "link_ipv6" $tuples_xtd)
    is_link_prefix_present=$(grep -c "link_ipv6_prefix" $tuples_xtd)

    is_router_addr_present=$(grep -c "router_ipv6" $tuples_xtd)
    is_router_prefix_present=$(grep -c "router_ipv6_prefix" $tuples_xtd)

    if [ ${is_link_addr_present:-0} -eq 0 -o ${is_link_prefix_present:-0} -eq 0 ]
    then
        echo "$loginfo fail to get link info for IPv6 Link Local" 1>>$log
    fi

    if [ ${is_router_addr_present:-0} -eq 0 -o ${is_router_prefix_present:-0} -eq 0 ]
    then
        echo "$loginfo fail to get IPv6 router assigned address" 1>>$log
    fi
fi

is_global_ip6_present=$(grep -c "global_ipv6" $tuples_xtd)
is_gateway_ip6_present=$(grep -c "global_ipv6" $tuples_xtd)
if [ ${is_global_ip6_present:-0} -gt 0 -a ${is_gateway_ip6_present:-0} -eq 0 ]
then
    echo "$loginfo fail to get IPv6 Gateway Address" 1>>$log
fi

####################################
#   Capturing firmware version
####################################
/usr/local/bin/collect_fw_vers.sh >>$tuples_to_filer

####################################
#    SEND TUPLES FILE
####################################
cp $tuples_to_filer $final_tuples || {
    echo "$loginfo fail to save $final_tuples" 1>>$log
    err=34
}


####################################
#    CLEAN UP TMP FILES
#############
#######################
rm -f $tmp_ifconfig_eth0
rm -f $tmp_ip_6_addr

echo "$loginfo $0 completed, err = $err" 1>>$log

# Once we get to this point just exit with good status for now.
#exit 0
exit $err
